<?php

if (!defined('ABSPATH')) {
  exit; // Exit if accessed directly.
}

/**
 * Chat assistant data shape:
 * [
 * 	name:"string",
 * 	description?:"string",
 * 	icon_url?:"string"
 * 	is_active?:"bool"
 * 	is_featured?:"bool".
 *	prompt:"string"
 * ].
 */
class Ultimate_AI_Chat_Assistant
{
  private $cpt_type = 'uai_cpt_assistant';
  private $meta_key = 'chat_assistant';
  static $ASSISTANT_VERSION_META_KEY = 'UAI_ASSISTANT_VERSION_META_KEY';
  static $ASSISTANT_VERSION = 2;

  public function __construct()
  {
  }

  public function is_new_version_available()
  {
    $pre_ver = get_option(Ultimate_AI_Chat_Assistant::$ASSISTANT_VERSION_META_KEY, false);

    if ($pre_ver == false || Ultimate_AI_Chat_Assistant::$ASSISTANT_VERSION > $pre_ver) {
      update_option(Ultimate_AI_Chat_Assistant::$ASSISTANT_VERSION_META_KEY, Ultimate_AI_Chat_Assistant::$ASSISTANT_VERSION);
      return true;
    }

    return false;
  }

  public function migrate_assistant_meta_to_new_structure()
  {
    $data = $this->ultimate_ai_get_chat_assistants();

    foreach ($data["chat_assistant"] as $assistant) {
      $meta = get_post_meta($assistant["id"], $this->meta_key, true);
      $meta["is_admin"] = "yes";
      $meta["is_active"] = "yes";
      update_post_meta($assistant["id"], $this->meta_key, $meta);
      update_post_meta($assistant["id"], "is_admin", "yes");
      update_post_meta($assistant["id"], "is_active", "yes");
    }
  }

  public function ultimate_ai_create_chat_assistant($name, $prompt, $desc = '', $icon_url = '', $id = 0)
  {

    if ($id == 0) {
      $chat_assistant_data = $this->ultimate_ai_get_chat_assistant_data_boilerplate();
    } else {
      $chat_assistant_data = get_post_meta($id, $this->meta_key, true);
    }

    $user = wp_get_current_user();

    $chat_assistant_data['name'] = $name;
    $chat_assistant_data['prompt'] = $prompt;
    if (!empty($desc)) $chat_assistant_data['description'] = $desc;
    if (!empty($icon_url)) $chat_assistant_data['icon_url'] = $icon_url;
    $chat_assistant_data['is_admin'] = current_user_can("edit_pages") ? "yes" : "no";



    $cpt_args = [
      'ID' => $id,
      'post_type' => $this->cpt_type,
      'post_author' => $user->ID,
      'post_content' => '',
      'post_title' => $name,
      'post_status' => 'publish',
      'meta_input' => [
        'chat_assistant' => $chat_assistant_data,
        'is_active' => $chat_assistant_data['is_active'],
        'is_admin' => $chat_assistant_data['is_admin'],
      ],
    ];
    $res = wp_insert_post($cpt_args);
    if (is_wp_error($res) || $res == 0) {
      return ['is_success' => false, 'message' => $res->get_error_message() ?? __('unknown message', ULTIMATE_AI_SLUG)];
    }

    $chat_assistant_data["id"] = $res;
    $chat_assistant_data['date'] = get_post_time('U', false, $res, false);
    return ['is_success' => true, 'data' => $chat_assistant_data];
  }

  public function ultimate_ai_delete_chat_assistant($chat_assistant_id)
  {
    restrict_req_for_admin();
    $is_success = wp_delete_post($chat_assistant_id);
    if (!$is_success) {
      return ['is_success' => false, 'message' => __('Chat Assistant deletion fail', ULTIMATE_AI_SLUG)];
    }

    return ['is_success' => true, 'data' => 'Chat Assistant deleted'];
  }

  public function ultimate_ai_get_chat_assistants($filter_array = [])
  {
    $args = [
      'post_type' => $this->cpt_type,
      'post_status' => 'publish',
      'posts_per_page' => -1, // all
    ];

    $args = array_merge($filter_array, $args);
    //var_dump(get_post_meta(505));
    //var_dump($args);

    $the_query = new WP_Query($args);
    $chat_assistants = [];

    if ($the_query->have_posts()) {
      while ($the_query->have_posts()) {
        $the_query->the_post();
        $chat_assistant = get_post_meta(get_the_ID(), $this->meta_key, true);
        $chat_assistant['date'] = get_post_time('U', false, get_the_ID(), false);
        $chat_assistant['id'] = get_the_ID();
        $chat_assistants[] = $chat_assistant;
      }
    }

    return ['chat_assistant' => $chat_assistants, 'total_chat_assistant' => $the_query->found_posts];
  }

  public function ultimate_ai_update_chat_assistant_meta($key_name, $value, $chat_id, $is_direct_meta = false)
  {
    restrict_req_for_admin();
    $chat_assistant = get_post_meta($chat_id, $this->meta_key, true);

    if (!$chat_assistant) {
      return ['is_success' => false, 'message' => __('Chat assistant meta not found', ULTIMATE_AI_SLUG)];
    }
    $chat_assistant[$key_name] = $value;
    $is_success = update_post_meta($chat_id, $this->meta_key, $chat_assistant);
    if ($is_direct_meta) {
      $is_success = update_post_meta($chat_id, $key_name, $value);
    }
    if (!$is_success) {
      return ['is_success' => false, 'message' => __('Chat assistant update fail', ULTIMATE_AI_SLUG)];
    }

    return ['is_success' => true, 'data' => $chat_assistant];
  }

  private function ultimate_ai_get_chat_assistant_data_boilerplate()
  {
    return [
      'name' => 'Unknown tool',
      'description' => 'Make your work faster with ai',
      'icon_url' => UAI_ASSETS_URL_IMG . 'tool_icons/default.svg',
      'is_active' => "yes",
      'is_admin' => "yes",
      'prompt' => '',
    ];
  }
}
